-- VHDL model for the Cypress CY7C1329 Synchronous-Pipelined SRAM
-- F J LaRosa 3/1/03

library ieee;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_1164.all;
use std.textio.all;
library work;
use work.txt_util.all;
use work.fjl_utils.all;

entity syncram is
	port (
	Clock : in std_logic;				-- Master clock, 30MHz
	ram_nADV : in std_logic;			-- advances address in a burst cycle
	ram_nADSP : in std_logic;			-- address strobe from Processor
	ram_nADSC : in std_logic;			-- address strobe from Controller
	ram_ADD : in std_logic_vector(15 downto 0);
	ram_nGW : in std_logic;			-- global Write Enable
	ram_nBWE : in std_logic;			-- Byte Write Enable
	ram_nBW : in std_logic_vector(3 downto 0);	-- Byte Write Select inputs
	ram_nCE1 : in std_logic;			-- Chip Enables
	ram_CE2 : in std_logic;
	ram_nCE3 : in std_logic;
	ram_nOE : in std_logic;			-- Output Enable
	ram_ZZ : in std_logic;				-- Sleep input
	ram_DQ : inout std_logic_vector(31 downto 0);
	ram_Mode : in std_logic
	);
end syncram;

architecture syncram1 of syncram is

-- Define the ram parameters:
constant RAMSIZE : integer := 127;
subtype ram_word is std_logic_vector(31 downto 0);	-- width of a ram word
type ram_block is array(0 to RAMSIZE) of ram_word;
signal ram_cell : ram_block;

signal all_ce_active : std_logic;						-- indicates all chip enables are active
signal all_ce_inactive : std_logic;						-- indicates device is not selected
signal read_cycle : std_logic;							-- indicates a read cycle
signal address_reg : std_logic_vector(15 downto 0);		-- internal address register
signal ram_state : std_logic;							-- ram state machine
signal read_data : std_logic_vector(31 downto 0);
signal read_in_progress : std_logic;					-- goes true a clock cycle after a read is initiated

begin							   
	
	-- All Chip Enables active:
	all_ce_active <= not(ram_nCE1) and ram_CE2 and not(ram_nCE3);
	all_ce_inactive <= ram_nCE1 and not(ram_CE2) and ram_nCE3;
	
	-- All Write enables inactive (to recognize a Read operation):
	read_cycle <= ram_nGW and ram_nBWE;
	
	-- Data port MUX:
	with ram_nOE select
	ram_DQ <= 	read_data after 5 ns when '0',
				"ZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZ" after 5 ns when others;
	
		
	-- Single Write via nADSP:
	process(Clock)
	begin						  
		if (all_ce_inactive = '1') then
			ram_state <= '0';
			read_in_progress <= '0';
			address_reg <= X"0000";
			read_data <= X"00000000";
		elsif rising_edge(Clock) then
			case ram_state is										
				
				when '0' =>				-- waiting for a chip enable
					
				if (((ram_nADSP = '0') or (ram_nADSC = '0')) and (all_ce_active = '1') and (read_cycle = '1')) then
					address_reg <= ram_ADD;		-- address reg gets latched on first clock edge
					read_in_progress <= '1';
					ram_state <= '0';
	
				elsif ((ram_nADSC = '0') and (ram_nADSP = '1') and (all_ce_active = '1')) then
				-- Single-clock-cycle Write...
					read_in_progress <= '0';
					if (ram_nGW = '0') then
						ram_cell(slv_to_int(ram_ADD, 16)) <= ram_DQ;
					elsif (ram_nBWE = '0') then
						if (ram_nBW(0) = '0') then
							ram_cell(slv_to_int(ram_ADD, 16))(7 downto 0) <= ram_DQ(7 downto 0);
						elsif (ram_nBW(1) = '0') then
							ram_cell(slv_to_int(ram_ADD, 16))(15 downto 8) <= ram_DQ(15 downto 8);
						elsif (ram_nBW(2) = '0') then
							ram_cell(slv_to_int(ram_ADD, 16))(23 downto 16) <= ram_DQ(23 downto 16);
						elsif (ram_nBW(3) = '0') then
							ram_cell(slv_to_int(ram_ADD, 16))(31 downto 24) <= ram_DQ(31 downto 24);
						end if;
					end if;				
					
				elsif ((ram_nADSP = '0') and (all_ce_active = '1')) then
				-- Two-clock-cycle Write...
					read_in_progress <= '0';
					address_reg <= ram_ADD;		-- address reg gets latched on first clock edge
					ram_state <= '1';
				else
					ram_state <= '0';
				end if;
				
				when others =>
				if (ram_nGW = '0') then
					ram_cell(slv_to_int(address_reg, 16)) <= ram_DQ;
				elsif (ram_nBWE = '0') then
					if (ram_nBW(0) = '0') then
						ram_cell(slv_to_int(address_reg, 16))(7 downto 0) <= ram_DQ(7 downto 0);
					elsif (ram_nBW(1) = '0') then
						ram_cell(slv_to_int(address_reg, 16))(15 downto 8) <= ram_DQ(15 downto 8);
					elsif (ram_nBW(2) = '0') then
						ram_cell(slv_to_int(address_reg, 16))(23 downto 16) <= ram_DQ(23 downto 16);
					elsif (ram_nBW(3) = '0') then
						ram_cell(slv_to_int(address_reg, 16))(31 downto 24) <= ram_DQ(31 downto 24);
					end if;
				end if;
				ram_state <= '0';
				
			end case;
			if (read_in_progress = '1') then
				read_data <= ram_cell(slv_to_int(address_reg, 16));
			end if;
		end if;
	end process;

	
end syncram1;
