--------------------------------------------------------------------------------
-- Company:        Draper Lab
-- Engineer:       F. J. LaRosa
--
-- Create Date:    15:28:33 10/22/06
-- Design Name:    
-- Module Name:    xnav - xnav1
-- Project Name:   
-- Target Device:  
-- Tool versions:  
-- Description:
--
-- Dependencies:
-- 
-- Revision:
-- Revision 0.01 - File Created
-- Additional Comments:
-- 
--------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

library UNISIM;
use UNISIM.VComponents.all;

entity xnav is
	port (
		nreset, clock : in std_logic;
		sin : in std_logic;
		sout : out std_logic;
		-- SDRAM IO:
		ram_clock : out std_logic;
		ba : out std_logic_vector(1 downto 0);
		a : out std_logic_vector(12 downto 0);
		cs_n, we_n, ras_n, cas_n, cke : out std_logic;
		dm : out std_logic_vector(1 downto 0);
		d : inout std_logic_vector(15 downto 0);
		dqs : inout std_logic_vector(1 downto 0)
	);
end xnav;

architecture xnav1 of xnav is

	-- Components:

	component IDELAY
		generic
		(
			IOBDELAY_TYPE : string := "FIXED";	-- (DEFAULT, FIXED, VARIABLE)
			IOBDELAY_VALUE : integer := 25		-- 80ps x 25 = 2 ns
		);
		port
		(
			O : out std_ulogic;
			C : in std_ulogic;
			CE : in std_ulogic;
			I : in std_ulogic;
			INC : in std_ulogic;
			RST : in std_ulogic
		);
	end component;

	component IDELAYCTRL
		port
		(
			RDY : out std_ulogic;
			REFCLK : in std_ulogic;
			RST : in std_ulogic
		);
	end component;

	component rs232_rcv is
		port (
			nreset, clock, en, sin : in std_logic;
			y : out std_logic_vector(7 downto 0);
			rdy, err : out std_logic
		);
	end component;

	-- Sequencer Update Rate:
	constant	SEQ_DIV : integer := 4;	-- 100MHz / 5 = 20 MHz Sequencer Update Rate

	-- This one is for simulation only (very fast):
	constant	BAUD_DIV : integer := 3;	-- 100MHz / 4 = 25 MHz 16X clock (25/16 = 1.5625 Mbps)
	-- This one is for board test mode - using UART port - at ~115.2kbaud:
	-- 100MHz / 54 = 1.852 MHz 16X clock (1.852/16 = 115.7 Kbps)
--	constant	BAUD_DIV : integer := 53;

	constant	ONWAIT : std_logic_vector	:= X"003f";	-- use for simulation
--	constant	ONWAIT : std_logic_vector	:= X"7d0f";	-- use for real hardware
	constant	RESWAIT : std_logic_vector	:= X"001f";	-- use for real simulation
--	constant	RESWAIT : std_logic_vector	:= X"00c8";	-- use for real hardware

	constant		CC_READ	: std_logic_vector := "00";
	constant		CC_WRITE	: std_logic_vector := "01";
	constant		CC_TRIG	: std_logic_vector := "10";
	constant		CC_RSRVD	: std_logic_vector := "11";

	constant		SS_SEQ	: std_logic_vector := "000";
	constant		SS_THERM	: std_logic_vector := "000";
	constant		SS_HK		: std_logic_vector := "000";
	constant		SS_PIX	: std_logic_vector := "000";

	constant		CH_HSEQ_BLOCK		: std_logic_vector := "0000000";
	constant		CH_HSEQ_ADDR		: std_logic_vector := "0000001";
	constant		CH_HSEQ_DATA		: std_logic_vector := "0000010";
	constant		CH_HSEQ_CONTROL	: std_logic_vector := "0000011";
	constant		CH_LSEQ_SEGMENT	: std_logic_vector := "0000100";
	constant		CH_LSEQ_ADDR		: std_logic_vector := "0000101";
	constant		CH_LSEQ_DATA		: std_logic_vector := "0000110";

	-- SDRAM Command encodings:
	constant	IDLE				: std_logic_vector := "1111";
	constant	NOP				: std_logic_vector := "0111";
	constant	ACTIVE			: std_logic_vector := "0011";
	constant	READ				: std_logic_vector := "0101";
	constant	WRITE				: std_logic_vector := "0100";
	constant	PRECHARGE		: std_logic_vector := "0010";
	constant	AUTOREFRESH		: std_logic_vector := "0001";
	constant	LOADMODE			: std_logic_vector := "0000";

	constant A_ZERO				: std_logic_vector := "0000000000000";
	constant A10_ONE				: std_logic_vector := "0010000000000";
	-- CAS Latencyc = 010, Burst Type = 0, Burst Length = 001:
	constant MODE_NORMAL			: std_logic_vector := "0000000100001"; 
	constant MODE_DLL_RESET		: std_logic_vector := "0000100100001";
	constant EMODE_DLL_ON		: std_logic_vector := "0000000000000";
	constant EMODE_DLL_OFF		: std_logic_vector := "0000000000001";

	signal	reset : std_logic;
	signal	seq_count : std_logic_vector(2 downto 0);
	signal	seq_tick : std_logic;
	signal	baud_count : std_logic_vector(6 downto 0);
	signal	baud_tick : std_logic;
	signal	sina, sinb : std_logic;
	signal	rsin_data : std_logic_vector(7 downto 0);
	signal	rsin_rdy, rsin_err : std_logic;

	type		cstates is (idle_st, read_st, write_st, write_arg, done);
	signal	cstate : cstates;
	signal	cmd_code, args, arg_count : std_logic_vector(1 downto 0);
	signal	subsystem : std_logic_vector(2 downto 0);
	signal	cmd_channel : std_logic_vector(6 downto 0);
	signal	write_data, read_data : std_logic_vector(20 downto 0);
	signal	go_write, go_read : std_logic;
	signal	read_args : std_logic_vector(1 downto 0);
	signal	stat_req : std_logic;

	signal	seq_running : std_logic;
	signal	hseq_block_reg, hseq_addr_reg : std_logic_vector(20 downto 0);
	signal	hseq_wdata_rega, hseq_wdata_regb : std_logic_vector(15 downto 0);
	signal	hseq_rdata_rega, hseq_rdata_regb : std_logic_vector(15 downto 0);
	signal	word_cnt : std_logic;
	signal	hseq_loop : std_logic;
	signal	lseq_seg_reg : std_logic_vector(9 downto 0);
	signal	lseq_addr_reg : std_logic_vector(13 downto 0);
	signal	lseq_wdata_reg : std_logic_vector(17 downto 0);
	signal	sdr_write, sdr_read, sdr_outen, sdr_readen : std_logic;

	type		sdr_states is (
					wait200us, cke_st, pa1, wait4a, emode, wait4b, pa2, wait4c,
					af1, wait16a, af2, wait16b, mode1, wait16c, idle_st,
					w_active, r_active, write_st, read_st
				);
	signal	sdr_state : sdr_states;
	signal	ba_a : std_logic_vector(1 downto 0);
	signal	a_a : std_logic_vector(12 downto 0);
	signal	wait_count : std_logic_vector(15 downto 0);
	signal	sdr_cmd : std_logic_vector(3 downto 0);
	signal	hseq_inc, rburst_done, rburst_done2 : std_logic;
	signal	d_outen, dqs_en : std_logic;

	signal	d_del : std_logic_vector(15 downto 0);
	type 		sdr2_states is (st0, st1, st2, st3, st4, st5, st6);
	signal	sdr2_state : sdr2_states;
	signal	sdr_wdata : std_logic_vector(15 downto 0);

begin

	reset	<= not(nreset);
	seq_running	<= '1';	-- temporary

	RCLK : process(nreset, clock)
	begin
		if (nreset = '0') then
			ram_clock	<= '0';
		elsif clock'event then
			if (clock = '1') then
				ram_clock	<= '1';
			else
				ram_clock	<= '0';
			end if;
		end if;
	end process;

	SYNC : process(nreset, clock)
	begin
		if (nreset = '0') then
			sina		<= '1';
			sinb		<= '1';
		elsif rising_edge(clock) then
			sina		<= sin;
			sinb		<= sina;
		end if;
	end process;

	-- Sequencer Update tick:
	SEQT : process(nreset, clock)
	begin
		if (nreset = '0') then
			seq_count	<= (others => '0');
			seq_tick	<= '0';
		elsif rising_edge(clock) then
			if (seq_count = SEQ_DIV) then
				seq_count	<= (others => '0');
				seq_tick	<= '1';
			else
				seq_count	<= seq_count + 1;
				seq_tick	<= '0';
			end if;
		end if;
	end process;

	-- Baud Rate generator:
	B16X : process(nreset, clock)
	begin
		if (nreset = '0') then
			baud_count	<= (others => '0');
			baud_tick	<= '0';
		elsif rising_edge(clock) then
			if (baud_count = BAUD_DIV) then
				baud_count	<= (others => '0');
				baud_tick	<= '1';
			else
				baud_count	<= baud_count + 1;
				baud_tick	<= '0';
			end if;
		end if;
	end process;

	RSIN : rs232_rcv
		port map (
			nreset => nreset,
			clock => clock,
			en => baud_tick,
			sin => sinb,
			y => rsin_data,
			rdy => rsin_rdy,
			err => rsin_err
		);

	-- Command Handler:
	-- Decodes:
	cmd_code		<= rsin_data(6 downto 5);
	-- State Machine:
	CMD : process(nreset, clock)
	begin
		if (nreset = '0') then
			args			<= (others => '0');
			subsystem	<= (others => '0');
			cmd_channel	<= (others => '0');
			arg_count	<= (others => '0');
			write_data	<= (others => '0');
			go_write		<= '0';
			go_read		<= '0';
			cstate		<= idle_st;
		elsif rising_edge(clock) then
			case cstate is
				when idle_st =>
					if (rsin_rdy = '1') then
						if (cmd_code = CC_READ) then
							args			<= rsin_data(4 downto 3);	--actually don't need
							arg_count	<= rsin_data(4 downto 3);
							subsystem	<= rsin_data(2 downto 0);
							cstate		<= read_st;
						elsif (cmd_code = CC_WRITE) then
							args			<= rsin_data(4 downto 3);
							arg_count	<= rsin_data(4 downto 3);
							subsystem	<= rsin_data(2 downto 0);
							cstate		<= write_st;
						elsif (cmd_code = CC_TRIG) then
							cstate	<= idle_st;
						else
							cstate	<= idle_st;
						end if;
					else
						cstate	<= idle_st;
					end if;

				when write_st =>
					if (rsin_rdy = '1') then
						cmd_channel	<= rsin_data(6 downto 0);
						if (args > 0) then
							cstate	<= write_arg;
						else
							cstate	<= idle_st;	-- temp;
						end if;
					else
						cstate	<= write_st;
					end if;

				when write_arg =>
					if (rsin_rdy = '1') then
						if (args = 1) then
							-- Will only be the first 7 bits..
							write_data(20 downto 14)	<= "0000000";
							write_data(13 downto 7)		<= "0000000";
							write_data(6 downto 0)		<= rsin_data(6 downto 0);
							go_write	<= '1';
							cstate	<= done;
						elsif (args = 2) then
							-- Will be the middle 7 bits...
							if (arg_count = 2) then
								write_data(20 downto 14)	<= "0000000";
								write_data(13 downto 7)		<= rsin_data(6 downto 0);
								arg_count	<= arg_count - 1;
								cstate	<= write_arg;
							-- Then the low 7 bts..
							elsif (arg_count = 1) then
								write_data(6 downto 0)		<= rsin_data(6 downto 0);
								go_write	<= '1';
								cstate	<= done;
							end if;
						else	-- (args = 3)
							-- Will be the upper 7 bits..
							if (arg_count = 3) then
								write_data(20 downto 14)		<= rsin_data(6 downto 0);
								arg_count	<= arg_count - 1;
								cstate	<= write_arg;
							-- Then the middle 7 bits..
							elsif (arg_count = 2) then
								write_data(13 downto 7)		<= rsin_data(6 downto 0);
								arg_count	<= arg_count - 1;
								cstate	<= write_arg;
							-- Then the lower 7 bits..
							elsif (arg_count = 1) then
								write_data(6 downto 0)	<= rsin_data(6 downto 0);
								go_write	<= '1';
								cstate	<= done;
							end if;
						end if;
					else
						cstate	<= write_arg;
					end if;

				when read_st =>
					if (rsin_rdy = '1') then
						cmd_channel	<= rsin_data(6 downto 0);
						go_read	<= '1';
						cstate	<= done;
					else
						cstate	<= read_st;
					end if;

				when done =>
					go_write	<= '0';
					go_read	<= '0';
					cstate	<= idle_st;

				when others =>
					cstate	<= idle_st;
			end case;
		end if;
	end process;

	-- Registers:
	REG : process(nreset, clock)
	begin
		if (nreset = '0') then
			rburst_done2		<= '0';
			hseq_block_reg		<= (others => '0');
			hseq_addr_reg		<= (others => '0');
			hseq_wdata_rega	<= (others => '0');
			hseq_wdata_regb	<= (others => '0');
			word_cnt			<= '0';
			hseq_loop			<= '0';
			lseq_seg_reg		<= (others => '0');
			lseq_addr_reg		<= (others => '0');
			lseq_wdata_reg		<= (others => '0');
			sdr_write			<= '0';
			sdr_read				<= '0';
			read_data			<= (others => '0');
			read_ss				<= "000";
			read_args			<= "00";
			stat_req				<= '0';
		elsif rising_edge(clock) then
			rburst_done2	<= rburst_done;	-- delayed pulse
			if (hseq_inc = '1') then	-- input from SDR2 state machine
				hseq_addr_reg	<= hseq_addr_reg + 2;	-- auto-inc after write or read burst
			end if;
			if (rburst_done = '1') then	-- input from SDR2 state machine
				read_data	<=	"00000" & hseq_rdata_rega;
				read_ss		<= "000";
				read_args	<= "11";
				stat_req		<= '1';
			elsif (rburst_done2 = '1') then
				read_data	<= "00000" & hseq_rdata_regb;
				read_ss		<= "000";
				read_args	<= "11";
				stat_req		<= '1';
			else
				stat_req		<= '0';
			end if;
			if (sdr_read = '1') then
				sdr_read	<= '0';			-- form a pulse
			end if;

			if (go_write = '1') then
				if (subsystem = SS_SEQ) then
					case cmd_channel is
						when CH_HSEQ_BLOCK =>
							hseq_block_reg	<= write_data;
						when CH_HSEQ_ADDR =>
							hseq_addr_reg	<= write_data;
						when CH_HSEQ_DATA =>
							if (word_cnt = '0') then
								hseq_wdata_rega	<= write_data(15 downto 0);
								word_cnt	<= '1';
							else						-- this is the second write of two
								hseq_wdata_regb	<= write_data(15 downto 0);
								word_cnt	<= '0';
								sdr_write	<= '1';
							end if;
						when CH_HSEQ_CONTROL =>
							hseq_loop		<= write_data(0);
						when CH_LSEQ_SEGMENT =>
							lseq_seg_reg	<= write_data(9 downto 0);
						when CH_LSEQ_ADDR =>			
							lseq_addr_reg	<= write_data(13 downto 0);
						when CH_LSEQ_DATA =>
							lseq_wdata_reg	<= write_data(17 downto 0);
						when others =>
							null;
					end case;

				elsif (subsystem = SS_THERM) then
					null;

				elsif (subsystem = SS_HK) then
					null;

				elsif (subsystem = SS_PIX) then
					null;

				end if;

			elsif (go_read = '1') then
				if (subsystem = SS_SEQ) then
					case cmd_channel is
						when CH_HSEQ_BLOCK =>
							read_data	<=	hseq_block_reg;
							read_args	<= "11";
							stat_req		<= '1';
						when CH_HSEQ_ADDR =>
							read_data	<=	hseq_addr_reg;
							read_args	<= "11";
							stat_req		<= '1';
						when CH_HSEQ_DATA =>
							if (word_cnt = '0') then
								word_cnt	<= '1';
								sdr_read	<= '0';
							else						-- this is the second read of two)
								word_cnt	<= '0';
								sdr_read	<= '1';	-- So we now request a read burst (of two)
							end if;
						when CH_HSEQ_CONTROL =>
							read_data	<=	X"0000" & "000" & seq_running& hseq_loop;
							read_args	<= "01";
							stat_req		<= '1';
						when CH_LSEQ_SEGMENT =>
							read_data	<=	X"00" & "000" & lseq_seg_reg;
							read_args	<= "10";
							stat_req		<= '1';
						when CH_LSEQ_ADDR =>			
							read_data	<=	X"0" & "000" & lseq_addr_reg;
							read_args	<= "10";
							stat_req		<= '1';
						when CH_LSEQ_DATA =>
							-- this should actually be the OUTPUT from the LSEQ memory:
							read_data	<= "000" & lseq_wdata_reg(17 downto 0);
							read_args	<= "11";
							stat_req		<= '1';
						when others =>
							null;
					end case;

				elsif (subsystem = SS_THERM) then
					null;

				elsif (subsystem = SS_HK) then
					null;

				elsif (subsystem = SS_PIX) then
					null;

				end if;

			else
				sdr_write	<= '0';
			end if;
		end if;
	end process;

	-- SDRAM State Machine:
	SDRAM : process(nreset, clock)
	begin
		if (nreset = '0') then
			cke			<= '0';
			ba_a			<= "00";		-- only using Bank 0 (also 00 means Mode Reg)
			ba				<= "00";
			a_a			<= A_ZERO;
			a				<= A_ZERO;
			sdr_cmd		<= IDLE;
			wait_count	<= ONWAIT;		
			sdr_outen	<= '0';
			dm				<= "00";
			sdr_state	<= wait200us;
		elsif rising_edge(clock) then
			case sdr_state is

				when wait200us =>
					if (wait_count = 0) then
						sdr_cmd		<= NOP;
						cke			<= '1';
						sdr_state	<= cke_st;
					else
						wait_count	<= wait_count - 1;
						sdr_state	<= wait200us;
					end if;

				when cke_st =>
					a_a			<= A10_ONE;			-- Pre-charge All
					sdr_cmd		<= PRECHARGE;
					sdr_state	<= pa1;
					
				when pa1 =>
					a_a			<= A_ZERO;
					sdr_cmd		<= NOP;
					wait_count	<= X"0003";
					sdr_state	<= wait4a;

				when wait4a =>
					if (wait_count = 0) then
						ba_a			<= "01";
						a_a			<= EMODE_DLL_ON;
						sdr_cmd		<= LOADMODE;
						sdr_state	<= emode;
					else
						wait_count	<= wait_count - 1;
						sdr_state	<= wait4a;
					end if;

				when emode =>
					ba_a			<= "00";
					a_a			<= MODE_DLL_RESET;
					sdr_cmd		<= LOADMODE;
					wait_count	<= RESWAIT;
					sdr_state	<= wait4b;

				when wait4b =>
					sdr_cmd		<= NOP;
					if (wait_count = 0) then
						a_a			<= A10_ONE;			-- Pre-charge All
						sdr_cmd		<= PRECHARGE;
						sdr_state	<= pa2;
					else
						wait_count	<= wait_count - 1;
						sdr_state	<= wait4b;
					end if;

				when pa2 =>
					a_a			<= A_ZERO;
					sdr_cmd		<= NOP;
					wait_count	<= X"0003";
					sdr_state	<= wait4c;

				when wait4c =>
					if (wait_count = 0) then
						a_a			<= A_ZERO;
						sdr_cmd		<= AUTOREFRESH;
						sdr_state	<= af1;
					else
						wait_count	<= wait_count - 1;
						sdr_state	<= wait4c;
					end if;

				when af1 =>
					sdr_cmd		<= NOP;
					wait_count	<= X"000f";
					sdr_state	<= wait16a;

				when wait16a =>
					if (wait_count = 0) then
						sdr_cmd		<= AUTOREFRESH;
						sdr_state	<= af2;
					else
						wait_count	<= wait_count - 1;
						sdr_state	<= wait16a;
					end if;

				when af2 =>
					sdr_cmd		<= NOP;
					wait_count	<= X"000f";
					sdr_state	<= wait16b;

				when wait16b =>
					if (wait_count = 0) then
						a_a			<= MODE_NORMAL;
						sdr_cmd		<= LOADMODE;
						sdr_state	<= idle_st;
					else
						wait_count	<= wait_count - 1;
						sdr_state	<= wait16b;
					end if;

				when idle_st =>
					a_a			<= A_ZERO;
					sdr_cmd		<= NOP;			-- (or should this be IDLE?)
					if (sdr_write = '1') then
						ba_a			<= "00";
						a_a			<= '0' & hseq_addr_reg(20 downto 9);	-- ROW
						wait_count	<= X"0002";
						sdr_cmd		<= ACTIVE;
						sdr_state	<= w_active;
					elsif (sdr_read	= '1') then
						ba_a			<= "00";
						a_a			<= '0' & hseq_addr_reg(20 downto 9);	-- ROW
						wait_count	<= X"0002";
						sdr_cmd		<= ACTIVE;
						sdr_state	<= r_active;
					else
						sdr_state	<= idle_st;
					end if;

				when w_active =>
					ba_a			<= "00";
					sdr_cmd		<= NOP;
					if (wait_count = 0) then
						a_a	<= "0000" & hseq_addr_reg(8 downto 0);		-- COLUMN
						wait_count	<= X"0002";
						sdr_cmd		<= WRITE;
						sdr_state	<= write_st;
					else
						wait_count	<= wait_count - 1;
						sdr_state	<= w_active;
					end if;

				when write_st =>
					sdr_outen	<= '1';
					sdr_cmd		<= NOP;
					if (wait_count = 0) then
						sdr_outen	<= '0';
						sdr_state	<= idle_st;
					else
						wait_count	<= wait_count - 1;
						sdr_state	<= write_st;
					end if;

				when r_active =>
					ba_a			<= "00";
					sdr_cmd		<= NOP;
					if (wait_count = 0) then
						a_a	<= "0000" & hseq_addr_reg(8 downto 0);		-- COLUMN
						wait_count	<= X"0002";
						sdr_cmd		<= READ;
						sdr_state	<= read_st;
					else
						wait_count	<= wait_count - 1;
						sdr_state	<= r_active;
					end if;

				when read_st =>
					sdr_readen	<= '1';
					sdr_cmd		<= NOP;
					if (wait_count = 0) then
						sdr_outen	<= '0';
						sdr_state	<= idle_st;
					else
						wait_count	<= wait_count - 1;
						sdr_state	<= read_st;
					end if;


				when others =>
					sdr_state	<= idle_st;
			end case;
		elsif falling_edge(clock) then
			ba		<= ba_a;
			a		<= a_a;
		end if;
	end process;

	cs_n	<=	sdr_cmd(3);
	ras_n	<=	sdr_cmd(2);	
	cas_n	<=	sdr_cmd(1);
	we_n	<=	sdr_cmd(0);

	-- Data and DFQS manager - needs to operate on both clock edges:
	SDR2 : process(nreset, clock)
	begin
		if (nreset = '0') then
			sdr_wdata			<= (others => '0');
			d_outen				<= '0';
			dqs_en				<= '0';
			hseq_inc				<= '0';
			rburst_done			<= '0';
			hseq_rdata_rega	<= (others => '0');
			hseq_rdata_regb	<= (others => '0');
			sdr2_state		<= st0;
		elsif clock'event then	-- using both edges
			case sdr2_state is

				when st0 =>
					if (sdr_outen = '1') then				-- (on a falling edge)
						sdr_wdata	<= hseq_wdata_rega;
						d_outen		<= '1';
						sdr2_state	<= st1;
					elsif (sdr_readen = '1') then
						sdr2_state	<= st4;					-- going to wait for DQS to assert..
					else
						sdr2_state	<= st0;
					end if;

				when st1 =>										-- (on a rising edge)
					sdr_wdata	<= hseq_wdata_regb;
					dqs_en	<= '1';						
					sdr2_state	<= st2;

				when st2 =>
					d_outen		<= '0';
					dqs_en		<= '0';
					hseq_inc		<= '1';
					sdr2_state	<= st3;
					
				when st3 =>
					hseq_inc		<= '0';
					sdr2_state	<= st0;

				when st4 =>
					if (dqs(0) = '1') then	-- (only need to watch one of them)
						hseq_rdata_rega	<= d_del;		-- grab the first word of the burst
						sdr2_state			<= st5;
					else
						sdr2_state			<= st4;
					end if;

				when st5 =>
					if (dqs(0) = '0') then
						hseq_rdata_regb	<= d_del;		-- grab the second word of the burst
						hseq_inc		<= '1';
						rburst_done	<= '1';
						sdr2_state			<= st6;
					else
						sdr2_state			<= st5;
					end if;

				when st6 =>
					hseq_inc		<= '0';
					rburst_done	<= '0';
					sdr2_state	<= st0;
					
				when others =>
					hseq_inc		<= '0';
					rburst_done	<= '0';
					sdr2_state	<= st0;

			end case;
		end if;
	end process;

	with d_outen select
		d	<=		sdr_wdata(15 downto 0) when '1',
					"ZZZZZZZZZZZZZZZZ" when others;

	with sdr_outen select
		dqs(1)	<=		dqs_en when '1',
							'Z' when others;

	with sdr_outen select
		dqs(0)	<=		dqs_en when '1',
							'Z' when others;

	IDEL0 : IDELAY
		port map
		(
			I => d(0),
			O => d_del(0),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL1 : IDELAY
		port map
		(
			I => d(1),
			O => d_del(1),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL2 : IDELAY
		port map
		(
			I => d(2),
			O => d_del(2),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL3 : IDELAY
		port map
		(
			I => d(3),
			O => d_del(3),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL4 : IDELAY
		port map
		(
			I => d(4),
			O => d_del(4),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL5 : IDELAY
		port map
		(
			I => d(5),
			O => d_del(5),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL6 : IDELAY
		port map
		(
			I => d(6),
			O => d_del(6),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL7 : IDELAY
		port map
		(
			I => d(7),
			O => d_del(7),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL8 : IDELAY
		port map
		(
			I => d(8),
			O => d_del(8),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL9 : IDELAY
		port map
		(
			I => d(9),
			O => d_del(9),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL10 : IDELAY
		port map
		(
			I => d(10),
			O => d_del(10),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL11 : IDELAY
		port map
		(
			I => d(11),
			O => d_del(11),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL12 : IDELAY
		port map
		(
			I => d(12),
			O => d_del(12),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL13 : IDELAY
		port map
		(
			I => d(13),
			O => d_del(13),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL14 : IDELAY
		port map
		(
			I => d(14),
			O => d_del(14),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

	IDEL15 : IDELAY
		port map
		(
			I => d(15),
			O => d_del(15),
			C => '0',
			CE => '0',
			INC => '0',
			RST => reset
		);

end xnav1;
