------------------------------------------------------------------
--  Event Driven CCD FPGA								                --
--  MIT Center for Space Research		   						    --
--	 								 	   				 		             --
--  Created by F. J. LaRosa 02-24-03							       --
--  LaRosa Engineering, Inc.									          --
------------------------------------------------------------------	
-- EDCCD is Event Driven CCD									          --
-- EDCCD is a Xilinx Spartan II design, and performs according	 --
-- to the document entitleD "Clock Sequencer for the EDCCD."	 --
------------------------------------------------------------------	
-- Modified 03-30-03 FJL										          --
-- 08-19-03 FJL nInit change to Init (positive true)			    --
------------------------------------------------------------------
-- Modified 02-16-05 FJL										          --
-- Inverted Init input; renamed to nInit						       --
-- Changed nACK & Busy timing per SPP style timing.				 --
--   Busy and nACK are both used for each byte write.			    --
--   If the FIFO is full, the Busy will stay high, and nACK		 --
--   nACK will be delayed until the FIFO is no longer full		 --	 								 	   				 		--
------------------------------------------------------------------	

library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.STD_LOGIC_unsigned.all;
use ieee.std_logic_arith.all;
library unisim;
use unisim.vcomponents.all;
	
entity edccd is
	port (
	Clock : in std_logic;					-- Master clock, 30MHz
	ExtClock : in std_logic;				-- should be the same 30MHz clock; used to make CLK & CLKn
	npoReset : in std_logic;				-- not Power-On Reset (onboard)
	
	-- Printer port signals:
	Din : in std_logic_vector(7 downto 0);	-- input data byte
	nStrobe	: in std_logic;					-- strobes the data in
	nSelect, nInit : in std_logic;			-- control input signals
	Busy, nAck, nError : out std_logic;		-- status outputs
	
	-- Synchronous-Pipeline Cache RAM interface lines:
	-- Note: the CLK input to the RAM will be connected to the board master clock,
	--		 which is the same 30MHz clock as this entity's "Clock."
	ram_nADV : out std_logic;			-- advances address in a burst cycle
	ram_nADSP : out std_logic;			-- address strobe from Processor
	ram_nADSC : out std_logic;			-- address strobe from Controller
	ram_ADD : out std_logic_vector(15 downto 0);
	ram_nGW : out std_logic;			-- global Write Enable
	ram_nBWE : out std_logic;			-- Byte Write Enable
	ram_nBW : out std_logic_vector(3 downto 0);	-- Byte Write Select inputs
	ram_nCE1 : out std_logic;			-- Chip Enable 1
	ram_CE2 : out std_logic;			-- Chip Enable 2
	ram_nCE3 : out std_logic;			-- Chip Enable 3
	ram_nOE : out std_logic;			-- Output Enable
	ram_ZZ : out std_logic;				-- Sleep input
	ram_Mode : out std_logic;			-- Mode = 0 is for linear burst mode (vs. interleaved)
	ram_DQ : inout std_logic_vector(31 downto 0);
	-- Additional note: the MODE pin should be tied LOW to enable
	--					linear burst mode (as opposed to interleaved mode.)
	
	-- Main outputs:
	CLK, CLKn : out std_logic;			--
	STRB, STRBn : out std_logic;		-- Complementary CLK outputs for output data "Dout"
	Active, Reset_out : out std_logic;	-- Active indicates data is valid, Reset_out is buffered Reset
	Dout : out std_logic_vector(31 downto 0)
	);
end edccd;

architecture behave1 of edccd is

-- fifo_16_255 is a 16 wide by 255 deep (not 256) FIFO, based on the
-- Spartan II RAMB4_S16_S16 Dual-Port RAM block:
component fifo_16_255 is
	port (clock			: IN std_logic;
		read_enable_in	: IN std_logic;
		write_enable_in	: IN std_logic;
		write_data_in	: IN std_logic_vector(15 downto 0);	-- changed from 8 to 16 FJL
		FIFO_gsr_in		: IN std_logic;
		read_data_out	: OUT std_logic_vector(15 downto 0);	-- changed from 8 to 16 FJL
		full_out		: OUT std_logic;
		empty_out		: OUT std_logic;
		fifocount_out	: OUT std_logic_vector(3 downto 0));
end component;

-- Constants:

-- UNCOMMENT THESE BEFORE IMPLEMENTING DEVICE:
--constant RAMSIZE : std_logic_vector(15 downto 0) := X"ffff";
-- Times for the ACK State Machine:
--constant ACK1 : std_logic_vector := "01111";	-- .533 uS
--constant ACK2 : std_logic_vector := "11111";	-- 1.07 uS

-- Test Values. Comment these out before implementing device:
constant RAMSIZE : std_logic_vector(15 downto 0) := X"001f";
-- Times for the ACK State Machine:
constant ACK1 : std_logic_vector := "00001";	--  66 nS
constant ACK2 : std_logic_vector := "00011";	-- 133 nS

-- Signals:			   

signal poReset_a, poReset_sync : std_logic;				-- synchronized poReset signals
signal Reset_a, Reset : std_logic;						-- FPGA master Reset signal

-- Printer port signals:
signal inStrobe_a, inStrobe_b, inStrobe_c : std_logic;	-- registered (and positive-true) nStrobe
signal inputStrobe1, inputStrobe2 : std_logic;			-- synchronized (and inverted) nStrobes
type ack_states is (aIdle, aInit, BusyUp, CheckFifo, AckOn);
signal ack_state : ack_states;
signal ack_counter : std_logic_vector(4 downto 0);		-- a monostable to time the nACK pulse
signal po_busy : std_logic;								-- asserts during boot-up
signal busy_counter : std_logic_vector(3 downto 0);		-- a monostable to delay the de-assertion of Busy
														--  until 8 Clocks have gone by after a Reset														-- and after the output clock DLL has locked
														-- (the clocks are stable)
signal fifo_busy : std_logic;							-- Busy dignal derived from fifo_full

-- Main regsters:
signal slr_reg : std_logic_vector(15 downto 0);			-- Sequencer Length Register
signal slr_reg_tc : std_logic_vector(15 downto 0);		-- Sequencer Length Register plus base address
signal cdr_reg : std_logic_vector(7 downto 0);			-- Clock Divider Register

-- Main state machine signals:
type main_states is
	(main_init, main_slrlo, main_slrhi, main_ramwait, main_ramwrite, 
	main_idle, main_active);
signal main_state : main_states;

-- RAM signals:
signal wram_add : std_logic_vector(15 downto 0);		-- RAM address generated by the main state machine (for writing).
signal wram_byte : std_logic_vector(1 downto 0);		-- Byte selector from main state machine.
signal wram_enable : std_logic;							-- RAM control signal from main state machine (for writing).
signal rram_enable : std_logic;							-- RAM control signal from the sequencer (for reading)
signal ram_byte_sel : std_logic_vector(2 downto 0);		-- Selector for a MUX that creates nWE for the SRAM.
signal write_data : std_logic_vector(31 downto 0);		-- Write data consisting of Din mapped to the appropriate byte position.
signal ram_oe_a, ram_oe_b, ram_oe_c : std_logic;		-- to create an extended OW signal for the SRAM

-- FIFO signals:
signal null_1 : std_logic;					   			-- Will be set to 0
signal null_16 : std_logic_vector(15 downto 0);			-- Will be set to X"0000"
signal fifo_empty, fifo_full : std_logic;				-- FIFO flags
signal fifo_wr_en, fifo_rd_en_cdr : std_logic;			-- FIFO enable lines
signal fifo_rd_en, fifo_rd_en2 : std_logic;				-- FIFO enable lines 
signal fifo_wr_add : std_logic_vector(15 downto 0);		-- FIFO Read address
signal fifo_rd_add : std_logic_vector(15 downto 0);		-- FIFO Write address
signal fifo_wr_data : std_logic_vector(15 downto 0);	-- FIFO Read data
signal fifo_rd_data : std_logic_vector(15 downto 0);	-- FIFO Write data
signal fifo_count : std_logic_vector(3 downto 0);		-- Upper nibble counter indicating roughly how full or empty
														-- the FIFO is - probably won't be used in this design.
-- Command receiver signals:
signal cmd_state : std_logic;							-- Small state machine for the command receiver
signal cmd_lo_byte : std_logic_vector(7 downto 0);		-- Low byte of a command, pre-stored
signal cmd_write_enable : std_logic;					-- Enables inputStrobe thru to the FIFO for writing
signal repeat_counter : std_logic_vector(9 downto 0);	-- counter number of times a command is repeated 
														--  terminal count is command(9 downto 0)
signal command : std_logic_vector(15 downto 0);			-- a command read from the FIFO. Nits 15:10 are upper 6 bits
														--  of the start addres (and lower 10 bits set to 0)
														--  Lower 10 bits of command are a repeat count
-- Sequencer signals:
signal cdr_tick, cdr_tick_del_a : std_logic;			-- Used for timing in the Sequencer
signal cdr_tick_del_b, cdr_tick_del_c, cdr_tick_del_d : std_logic;		-- ...
signal cdr_counter : std_logic_vector(7 downto 0);		-- Counts the number of clocks per Sequencer word
														--  terminal count is the CDR register.
signal cdr_pending : std_logic;
signal out_enable, out_enable_pend : std_logic;			-- out_enable is used as the Active signal
signal out_strobe : std_logic;							-- Used as the output strobe
signal out_counter : std_logic_vector(7 downto 0);		-- delayed "cdr" counter. This counter creates out_strobe
signal rram_add : std_logic_vector(15 downto 0);		-- RAM address generated by the Sequencer (for reading).
type seqen_states is
	(seq_off, seq_prep, seq_on, seq_off_pending); 
signal seqen_state : seqen_states;
signal get_cmd1 : std_logic;							-- enables reading the first command from the FIFO
signal seq_enable : std_logic;							-- enables the Sequencer
signal repeat_tc, slr_tc : std_logic;					-- terminal count decodes

signal test_count : std_logic_vector(3 downto 0);	
begin
	-- Port Maps:
		
	-- This is the FIFO
	FIFO0 : fifo_16_255
	port map (
		clock =>	Clock,
		read_enable_in => fifo_rd_en,
		write_enable_in => fifo_wr_en,
		write_data_in => fifo_wr_data,
		FIFO_gsr_in => Reset,
		read_data_out => fifo_rd_data,
		full_out =>	fifo_full,
		empty_out => fifo_empty,
		fifocount_out => fifo_count
	);

	-- Permananet assigments:
	nError <= '1';		-- Keep nError de-asserted until I can think of a use for it.
	ram_CE2 <= '1';		-- Assert CE2
	ram_nCE3 <= '0';	-- .. and nCE3 - then use nCE1 for all the work
	ram_nADV <= '0';	-- Won't be using ADV (burst operations)
	ram_nGW <= '1';		-- Disable Global write for now
	ram_Mode <= '0';	-- set the mode to 0 for linear burst operation
	ram_ZZ <= '0';		-- keep the SRAM awake (not sleep mode)

	CLK <= ExtClock;		-- Buffered version of Clock
	Clkn <= not(ExtClock);	-- Inverted version of Clock
	
	STRB <= out_strobe;
	STRBn <= not(out_strobe);
	Active <= out_enable;
	Reset_out <= Reset;		-- This mirrors both power-on-reset and printer-init reset
	
	-- Reset synchronization. poReset_sync is asynchronously asserted
	-- when the board reset asserts (poReset), and is synchronously de-asserted.
	-- Note: internal resets are positive-true.
	PORST: process(Clock, npoReset)
	begin
		if (npoReset = '0') then
			poReset_a <= '1';						-- first stage of piped npoReset
			poReset_sync <= '1';					-- synchronized npoReset
		elsif rising_edge(Clock) then
			poReset_a <= not(npoReset);
			poReset_sync <= poReset_a;
		end if;
	end process;

	-- This process generates the master Reset signal for the chip
	-- When printer port nSelect and nInit are both low, Reset asserts
	-- synchronously to the Clock. poReset_sync also apprears in this signal,
	-- via the first IF statement below:
	MASTERRST: process(Clock, poReset_sync)
	begin
		if (poReset_sync = '1') then
			Reset_a <= '1';
			Reset <= '1';
		elsif rising_edge(Clock) then
			Reset_a <= not(nSelect) and not(nInit);
			Reset <= Reset_a;
		end if;
	end process;		
	
	-- Synchronize nStrobe from the Printer port.
	-- nStrobe can be > 50uS, so we want to create a leading edge pulse at 30MHz.
	PPORT: process(Clock, Reset)
	begin
		if (Reset = '1') then
			inStrobe_a <= '0';					-- registered nStrobe (a, b, c)
			inStrobe_b <= '0';
			inStrobe_c <= '0';
		elsif (rising_edge(Clock)) then
			inStrobe_a <= not(nStrobe);		-- invert and then pipe nStrobe through
			inStrobe_b <= inStrobe_a;
			inStrobe_c <= inStrobe_b;
		end if;
	end process;
	
	-- Create pulses to mark the leading and trailing edges of the nStrobe input:
	inputStrobe1 <= inStrobe_b and not(inStrobe_c);	-- leading edge
	inputStrobe2 <= not(inStrobe_b) and inStrobe_c; -- trailing edge
	
	ACKSTATE: process(Clock, Reset)
	-- This state machine creates nACK and Busy strobes to acknowledge the host on the printer port
	begin
		if (Reset = '1') then
			nAck			<= '1';					-- de-assert					 
			fifo_busy	<= '0';					-- default to Not Busy
			ack_counter <= (others => '0');
			ack_state	<= aIdle;
		elsif rising_edge(Clock) then
			case ack_state is
				when aIdle =>
					if (inputStrobe2 = '1') then		-- An inputStrobe has occured, which also means data has been latched
						ack_state	<= aInit;
					else
						ack_state	<= aIdle;
					end if;

				when aInit =>
					if (ack_counter = ACK1) then
						ack_counter <= (others => '0');
						fifo_busy	<= '1';			-- assert Busy
						ack_state	<= BusyUp;
					else
						ack_counter <= ack_counter + 1;
						ack_state	<= aInit;
					end if;

				when BusyUp =>
					if (ack_counter = ACK1) then
						ack_counter <= (others => '0');
						ack_state	<= CheckFifo;
					else
						ack_counter <= ack_counter + 1;
						ack_state	<= BusyUp;
					end if;

				when CheckFifo =>
					-- Have to hold here if the FIFO is currently full.
					if (fifo_full = '0') then
						nACK		<= '0';			-- assert nACK
						ack_state	<= AckOn;
					else
						ack_state	<= CheckFifo;
					end if;

				when AckOn =>
					if (ack_counter = ACK2) then
						fifo_busy	<= '0';			-- de-assert Busy
						nACK		<= '1';			-- de-assert nACK
						ack_counter	<= (others => '0');
						ack_state	<= aIdle;
					else
						ack_counter <= ack_counter + 1;
						ack_state	<= AckOn;
					end if;

				when others =>
					null;
			end case;
		end if;
	end process;
	
	BUSYPROC: process(Clock, Reset)
	-- For now, this process keeps Busy asserted when in Reset, and then de-asserts Busy
	-- after 16 Clock pulses have gone by (dll locked is commented out right now). 
	-- This guarantees that the host on the printer port will not try to access this unit
	-- until after Xilinx configuration is complete and the clocks are stable.
	begin
		if (Reset = '1') then
			po_busy 			<= '1';			-- start out Busy
			busy_counter	<= X"0";			-- clear the counter
		elsif rising_edge(Clock) then
			if (busy_counter = X"f") then
				po_busy <= '0';				-- OK to clear Busy now
			else
				busy_counter <= busy_counter + 1;
			end if;
		end if;
	end process;
	
	Busy <= po_busy or fifo_busy;	
	
	-- Main State Machine:
	MAIN_SM: process(Clock, Reset)
	begin
		if (Reset = '1') then
			main_state <= main_init;		
			wram_add <= X"0000";				-- Initialize RAM write address to 0
			wram_byte <= "00";					-- Initialize RAM byte to 0 (of 3);
			wram_enable <= '0';					-- Default enable is off
			-- Registers:
			slr_reg <= X"0000";
			cdr_reg <= X"00";
		elsif rising_edge(Clock) then
			case main_state is
				
				when main_init =>					-- This is the starting state
				if (inputStrobe1 = '1') then
					slr_reg(7 downto 0) <= Din;		-- Load the Seq Length Register lo byte
					main_state <= main_slrlo;		-- Advance to main_slrlo state
				else
					main_state <= main_init;		-- else wait for the strobe
				end if;
				
				when main_slrlo =>					-- Just loaded SLR Low
				if (inputStrobe1 = '1') then
					slr_reg(15 downto 8) <= Din;	-- Load the Seq Length Register hi byte
					main_state <= main_slrhi;		-- Advance to main_slrhi state
				else
					main_state <= main_slrlo;		-- else wait for the strobe
				end if;
				
				when main_slrhi =>					-- Just loaded SLR High
				if (inputStrobe1 = '1') then
					cdr_reg <= Din;					-- Load the Clock Divider Register
					main_state <= main_ramwait;		-- Advance to the main_cdr state
				else
					main_state <= main_slrhi;		-- else wait for the strobe
				end if;
				
				when main_ramwait =>				-- CDR_Reg has just been latched
				if (inputStrobe1 = '1') then
					wram_enable <= '1';				-- Assert the enable (becomes ~adsc, we, etc.)
					main_state <= main_ramwrite;	-- Advance to main_ramwrite state
													-- wram_add and wram_byte already cleared by init
				else
					main_state <= main_ramwait;		-- else wait for the strobe
				end if;
				
				when main_ramwrite =>
				wram_enable <= '0';					-- De-assert the enable (so it's 1 clock cycle)
				if ((wram_add = RAMSIZE) and (wram_byte = "11")) then	-- we just wrote the very last byte
					main_state <= main_idle;							-- Go to Idle 
				elsif (wram_byte = "11") then		-- Just wrote byte 3 of the current address
					wram_byte <= "00";				-- byte wraps around to 0 (bytes 0, 1, 2, 3, 0, ...)
					wram_add <= wram_add + 1;		-- .. and the address increments
					main_state <= main_ramwait;		-- Go and wait for the next strobe.
				else
					wram_byte <= wram_byte + 1;		-- else increment the byte
					main_state <= main_ramwait;		-- Go and wait for the next strobe.
				end if;
				
				when main_idle =>					-- All RAM has been loaded...
				if (fifo_empty = '0') then			-- Wait for the FIFO to contain a command...
					main_state <= main_active;		-- If so, go to the Active state
				else
					main_state <= main_idle;		-- If FIFO empty, just wait here
				end if;
				
				when main_active =>					-- FIFO is not empty (commands available)
				if (fifo_empty = '1') then			-- If FIFO empties out,
					main_state <= main_idle;		-- go back to Idle...
				else
					main_state <= main_active;		-- otherwise stay in Active
				end if;
			end case;
	   end if;
	end process;
	
	-- Assign the chip select:
	ram_nCE1 <= not(wram_enable or rram_enable);
	-- Assign the write strobe:
	ram_nADSC <= not(wram_enable);
	-- Assign the Byte Write Enable:
	ram_nBWE <= not(wram_enable);
	
	-- Assign the Write Byte identifier. First create a select input to the MUX,
	-- so that nBWE will be "1111" when we are not actually writing
	ram_byte_sel <= wram_enable & wram_byte;
	-- Next, the MUX:
	with ram_byte_sel select
	ram_nBW <=	"1110" when "100",
				"1101" when "101",
				"1011" when "110",
				"0111" when "111",
				"1111" when others;	 
	
	-- Assign the address port:
	with seq_enable select
		ram_ADD <=	rram_add when '1',
					wram_add when others;

	-- Create the RAM write data from Din and wram_byte:
	with wram_byte select
	write_data <=	X"000000" & Din when "00",
					X"0000" & Din & X"00" when "01",
					X"00" & Din & X"0000" when "10",
					Din & X"000000" when others;	-- (11)
	
	-- Tristate data port MUX:
	with wram_enable select
	ram_DQ <= 	write_data when '1',
				"ZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZ" when others;

	-- Command Receiver. This process is disabled during the Initialization.
	-- When the Main State Machine has entered main_idle, and as long as it
	-- is in either main_idle or main_active, this process will accept
	-- data from the printer port, package 2 bytes into a word and stuff it
	-- into the FIFO.
	CMD_RCV: process(Clock, Reset)
	begin
		if (Reset = '1') then
			cmd_state <= '0';				-- "accept lo cmd byte" state
			cmd_lo_byte <= X"00";
			cmd_write_enable <= '0';
		elsif rising_edge(Clock) then
			case cmd_state is
				when '0' =>
				if ((main_state = main_idle) or (main_state = main_active)) then
					if (inputStrobe1 = '1') then
						cmd_lo_byte(7 downto 0) <= Din;	-- Load the lo byte of the command word
						cmd_write_enable <= '1';		-- This will enable inputStrobe thru to the FIFO
						cmd_state <= '1';				-- Advance to the "accept hi cmd byte" state
					else
						cmd_state <= '0';
					end if;
				end if;
				
				when others =>						-- (state 1)
				-- In this state, the low byte has already been primed.
				-- When the Input Strobe asserts, the Din byte will be merged with the
				-- pre-stored cmd_byte and presented to the input to the FIFO.
				-- inputStrobe, having already been synced to a clock width,
				-- will be forward to the FIFO Read enable input to stuff in the 16-bit command.
				if (inputStrobe1 = '1') then			-- This time, the imput strobe will be
					cmd_write_enable <= '0';		-- De-assert the cmd write enable in transition back to state 0
					cmd_state <= '0';
				end if;
			end case;
		end if;
	end process;						 
	
	fifo_wr_data <= Din & cmd_lo_byte;					-- Assign the FIFO Write Data
	fifo_wr_en <= cmd_write_enable and inputStrobe1;		-- Assign the FIFO Write Enable
	
	SEQEN: process(Clock, Reset)
	-- This small state machine manages the enabling of the Sequencer (see process SEQ)
	begin
		if (Reset = '1') then 
			seqen_state <= seq_off;
			get_cmd1 <= '0';
			seq_enable <= '0';	   
		elsif rising_edge(Clock) then
			case seqen_state is
				
				when seq_off =>
				if (main_state = main_active) then	-- The main state machine must be in main_active
					seqen_state <= seq_prep;			-- Go to Prep
					get_cmd1 <= '1';						-- pulse to read the first cmd from the FIFO
				else
					seqen_state <= seq_off;
				end if;
				
				when seq_prep =>
				seqen_state <= seq_on;				-- Next state will be ON, after this 1-clock-cycle pulse
				get_cmd1 <= '0';						-- clear get_cmd1 pulse
				seq_enable <= '1';					-- The official Sequencer Enable signal
				
				when seq_on =>
				if (main_state /= main_active) then	-- Wait for the main state machine to drop out of main_active
					seqen_state <= seq_off_pending;	-- Go to a  pending" state, rather than just dropping seq_enable
				else								--  (in case a command has just come in)
					seqen_state <= seq_on;
				end if;
				
				when seq_off_pending =>
				if (main_state = main_active) then	-- if a command has come in, go back to ON
					seqen_state <= seq_on;
				elsif (fifo_rd_en = '1') then		-- otherwise go to OFF on the last fifo read pulse
					seqen_state <= seq_off;
					seq_enable <= '0';
				else
					seqen_state <= seq_off_pending;
				end if;
				
				when others =>						-- catch illegal states
				seqen_state <= seq_off;
				seq_enable <= '0';
			
			end case;
		end if;
	end process;
	
	CLKDIV: process(Clock, Reset)
	-- This process uses the CDR register to make pulses (ticks) that run the Sequencer
	-- This tick is called cdr_tick, and is used (inverted) as the nADSP RAM control signal
	-- to clock the read address into the RAM. The read data appears one clock cycle after
	-- the Clock rising edge that samples cdr_tick.
	-- The data latching timing is in another process, but this process does create a
	-- version of cdr_tick tat is 2 clocks delayed - called cdr_tick_del_b, and which is
	-- used to latch the data from the RAM.
	begin
		if (Reset = '1') then
			cdr_counter <= X"00";
			cdr_tick <= '0';
			cdr_tick_del_a <= '0';
			cdr_tick_del_b <= '0';
			cdr_tick_del_c <= '0';
			cdr_tick_del_d <= '0';
			cdr_pending <= '0';
		elsif rising_edge(Clock) then
			cdr_tick_del_a <= cdr_tick;			-- create a 2-clock delay of cdr_tick
			if (cdr_reg = X"00") then
				cdr_tick_del_c <= cdr_tick_del_b;
				cdr_tick_del_d <= cdr_tick_del_c;
				if (seq_enable = '1') then
					cdr_tick_del_b <= cdr_tick_del_a;	-- to latch the SRAM data
				else					
					cdr_tick_del_a <= '0';
					cdr_tick_del_b <= '0';
				end if;
				if ((cdr_tick_del_b = '0') and (cdr_tick_del_c = '1')) then
					cdr_pending <= '1';
				end if;
				if (cdr_pending = '1') and (cdr_tick_del_d = '0') then
					cdr_tick <= '0';
				end if;
			else
				cdr_tick_del_b <= cdr_tick_del_a;	-- to latch the SRAM data
				cdr_tick_del_c <= cdr_tick_del_b;
			end if;
			if ((seq_enable = '1') or (out_enable = '1')) then	
				if (cdr_counter = cdr_reg) then
					cdr_counter <= X"00";
					if (seq_enable = '1') then	-- ** redundant?
						cdr_tick <= '1';
					end if;
				else
					cdr_counter <= cdr_counter + 1;
					cdr_tick <= '0';
				end if;
			else
				cdr_counter <= X"00";				-- clear counter when seq_enable goes low
			end if;
		end if;
	end process;
	
	OUTSTROBE: process(Clock, Reset)
	-- This process creates the output latching timing and the Active signal.
	-- The output strobe is created in the process below, called OSTROBE.
	-- This process runs a counter that matches cdr_counter in frequency,
	-- but differs in phase. Also, this counter does not start up until
	-- the first cdr_tick_del_b has occurred. This is because the first
	-- cdr_tick_del_b will PRECEDE the first real data word, whereas what
	-- we want is a first strobe that occurs near the end of the first
	-- data word, and then continues from there.
	begin
		if (Reset = '1') then
			out_enable <= '0';
			out_enable_pend <= '0';
			out_counter <= X"00";	
		elsif rising_edge(Clock) then
			if (cdr_reg = X"00") then
				out_enable <= cdr_tick_del_b or cdr_tick_del_c;
			else
				if (cdr_tick_del_b = '1') then
					out_enable <= '1';					-- assert Active on the first incoming tick
					out_enable_pend <= '1';
				end if;
				if ((seq_enable = '0') and (out_counter = cdr_reg)) then
					 out_enable_pend <= '0';			-- this signal gives use an extra cycle at the end (if stopping)
				end if;						
				if ((out_enable_pend = '0') and (out_counter = cdr_reg)) then
					out_enable <= '0';					-- this is where we really stop
				end if;
				if (out_enable = '1') then
					if (out_counter = cdr_reg) then
						out_counter <= X"00";
					else
						out_counter <= out_counter + 1;
					end if;
				end if;
			end if;
		end if;
	end process;
	
	OSTROBE: process(out_enable, out_counter, cdr_reg, ExtClock, cdr_tick_del_c, cdr_tick_del_d)
	-- This process creates the out_strobe signal, which becomes Strobe.
	-- This pulse occurs at the rollover point of out_counter (see above process).
	begin
		if (cdr_reg = X"00") then
			if ((cdr_tick_del_c = '1') or (cdr_tick_del_d = '1')) then
				out_strobe <= not(ExtClock);	-- need to use the clock, but it's illegal to use Clock
												-- as an input to a logic gate, so use the copy
			end if;
		elsif (out_enable = '1') then
			if (out_counter = cdr_reg) then
				out_strobe <= '1';
			else
				out_strobe <= '0';
			end if;
		else
			out_strobe <= '0';
		end if;
	end process;
	
	SEQ: process(Clock, Reset)
	-- This process is the main Sequencer, and controls the FIFO read enable signal
	-- and a counter which serves as a Read Address counter. Terminal Count decodes
	-- appear in a separate, non-registered process below (TC).
	begin
		if (Reset = '1') then			   
			rram_add <= X"0000";			
			command <= X"0000";
			repeat_counter <= "0000000000";
			slr_reg_tc <= X"0000";
		elsif rising_edge(Clock) then
			-- This IF construct handles the reading of the first command when the
			-- sequencer is first starting up:
			if (fifo_rd_en2 = '1') then
				command <= fifo_rd_data;		-- we latched the command in case of a command repeat
				--... and latch the actual read address at the sae time:
				rram_add <= fifo_rd_data(15 downto 10) & "0000000000";
				-- Create a Seq Length terminal count by adding the base address in the command:
				slr_reg_tc <= (fifo_rd_data(15 downto 10) & "0000000000") + slr_reg;
				rram_add <= fifo_rd_data(15 downto 10) & "0000000000";
			elsif (seq_enable = '1') then
				if (cdr_tick = '1') then									-- time this to the clkdiv tick
					if (slr_tc = '1') then			-- (see TC below)
						rram_add <= fifo_rd_data(15 downto 10) & "0000000000";	-- rollover
						if (repeat_tc = '1') then	-- (see below)
							repeat_counter <= "0000000000";					-- rollover
						else
							repeat_counter <= repeat_counter + 1;
						end if;
					else
						rram_add <= rram_add + 1;
					end if;
				end if;
			end if;
		end if;
	end process;					
	
	TC: process(repeat_counter, command, rram_add, slr_reg_tc)
	-- Decode the terminal counts for the Repeat Counter and the SLR Counter
	begin
		if (repeat_counter = command(9 downto 0)) then
			repeat_tc <= '1';
		else
			repeat_tc <= '0';
		end if;
		if ((rram_add = slr_reg_tc) or (rram_add = X"ffff")) then	
			slr_tc <= '1';
		else
			slr_tc <= '0';
		end if;
	end process;
	
	-- Create the FIFO Read Enable signal
	fifo_rd_en <= (repeat_tc and slr_tc and cdr_tick) or get_cmd1;
	
	-- Create a pulse delayed one clock from fifo_rd_en:
	FRE : process(Clock, Reset)
	begin
		if (Reset = '1') then
			fifo_rd_en2 <= '0';
		elsif rising_edge(Clock) then
			fifo_rd_en2 <= fifo_rd_en;
		end if;
	end process;
	
	-- Assign SRAM Readl control signals:		   
	ram_nADSP <= not(cdr_tick);
	
	RAMOE: process(Clock, Reset)
	-- This process creates the RAM Output Enable signal
	begin
		if (Reset = '1') then
			ram_oe_a <= '0';
			ram_oe_b <= '0';
			ram_oe_c <= '0';
		elsif rising_edge(Clock) then
			ram_oe_a <= seq_enable or out_enable;
			ram_oe_b <= ram_oe_a;
			ram_oe_c <= ram_oe_b;
		end if;
	end process;
	ram_nOE <= not (ram_oe_a or ram_oe_b or ram_oe_c);
	rram_enable <= ram_oe_a or ram_oe_b or ram_oe_c;
	
	DOUTLATCH: process(Clock, Reset)
	-- This process is the dta output latch.
	begin
		if (Reset = '1') then
			Dout <= X"00000000";
		elsif rising_edge(Clock) then
			if (cdr_reg = X"00") then
				if ((cdr_tick_del_b = '1') or (cdr_tick_del_c = '1')) then
					Dout <= ram_DQ;
				end if;
			elsif (cdr_tick_del_b = '1') then
				Dout <= ram_DQ;
			end if;
		end if;
	end process;
	
end behave1;

	


